<?php
declare(strict_types=1);

/**
 * CakePHP(tm) : Rapid Development Framework (http://cakephp.org)
 * Copyright (c) Cake Software Foundation, Inc. (http://cakefoundation.org)
 *
 * Licensed under The MIT License
 * For full copyright and license information, please see the LICENSE.txt
 * Redistributions of files must retain the above copyright notice.
 *
 * @copyright     Copyright (c) Cake Software Foundation, Inc. (http://cakefoundation.org)
 * @link          http://cakephp.org CakePHP(tm) Project
 * @license       http://www.opensource.org/licenses/mit-license.php MIT License
 */
namespace Authentication\PasswordHasher;

use Cake\Core\App;
use RuntimeException;

/**
 * Builds password hashing objects
 */
class PasswordHasherFactory
{
    /**
     * Returns password hasher object out of a hasher name or a configuration array
     *
     * @param string|array $passwordHasher Name of the password hasher or an array with
     * at least the key `className` set to the name of the class to use
     * @return \Authentication\PasswordHasher\PasswordHasherInterface Password hasher instance
     * @throws \RuntimeException If password hasher class not found or it does
     *   not implement \Authentication\PasswordHasher\PasswordHasherInterface
     */
    public static function build($passwordHasher): PasswordHasherInterface
    {
        $config = [];
        if (is_string($passwordHasher)) {
            $class = $passwordHasher;
        } else {
            $class = $passwordHasher['className'];
            $config = $passwordHasher;
            unset($config['className']);
        }

        $className = App::className($class, 'PasswordHasher', 'PasswordHasher');
        if ($className === null) {
            throw new RuntimeException(sprintf('Password hasher class `%s` was not found.', $class));
        }

        $hasher = new $className($config);
        if (!($hasher instanceof PasswordHasherInterface)) {
            throw new RuntimeException('Password hasher must implement PasswordHasherInterface.');
        }

        return $hasher;
    }
}
